<?php
/**
 * Meta boxes functions for the plugin.
 *
 * @package    Theme_Junkie_Deals
 * @since      0.1.0
 * @author     Theme Junkie
 * @copyright  Copyright (c) 2016, Theme Junkie
 * @license    http://www.gnu.org/licenses/gpl-2.0.html
 */

/* Register meta boxes. */
add_action( 'add_meta_boxes', 'tjd_add_meta_boxes' );

/* Save meta boxes. */
add_action( 'save_post', 'tjd_meta_boxes_save', 10, 2 );

/**
 * Registers new meta boxes for the 'team_item' post editing screen in the admin.
 *
 * @since  0.1.0
 * @access public
 * @link   http://codex.wordpress.org/Function_Reference/add_meta_box
 */
function tjd_add_meta_boxes() {

	/* Check if current screen is team page. */
	if ( 'deal' != get_current_screen()->post_type )
		return;

	add_meta_box( 
		'tjd-metaboxes-deal',
		__( 'Deal Settings', 'tjdeals' ),
		'tjd_metaboxes_display',
		'deal',
		'normal',
		'high'
	);

}

/**
 * Displays the content of the meta boxes.
 *
 * @param  object  $post
 * @since  0.1.0
 * @access public
 */
function tjd_metaboxes_display( $post ) {

	wp_nonce_field( basename( __FILE__ ), 'tjd-metaboxes-deal-nonce' ); ?>

	<div id="tjd-block">

		<div class="tjd-label">
			<label for="tjd-coupon-code">
				<strong><?php _e( 'Coupon Code', 'tjdeals' ); ?></strong><br />
				<span class="description"><?php _e( 'Coupon code for this deal.', 'tjdeals' ); ?></span>
			</label>
		</div>

		<div class="tjd-input">
			<input type="text" name="tjd-coupon-code" id="tjd-coupon-code" value="<?php echo sanitize_text_field( get_post_meta( $post->ID, 'tjd_coupon_code', true ) ); ?>" size="30" style="width: 99%;" placeholder="<?php esc_attr_e( 'THEMEJUNKIE2016', 'tjdeals' ); ?>" />
		</div>

	</div><!-- #tjd-block -->

	<div id="tjd-block">

		<div class="tjd-label">
			<label for="tjd-custom-url">
				<strong><?php _e( 'Destination URL', 'tjdeals' ); ?></strong><br />
				<span class="description"><?php _e( 'This could be your affiliate link etc.', 'tjdeals' ); ?></span>
			</label>
		</div>

		<div class="tjd-input">
			<input type="text" name="tjd-custom-url" id="tjd-custom-url" value="<?php echo esc_url( get_post_meta( $post->ID, 'tjd_custom_url', true ) ); ?>" size="30" style="width: 99%;" placeholder="<?php echo esc_attr( 'https://theme-junkie.com/' ); ?>" />
		</div>

	</div><!-- #tjd-block -->

	<div id="tjd-block">

		<div class="tjd-label">
			<label for="tjd-button-text">
				<strong><?php _e( 'Button Text', 'tjdeals' ); ?></strong><br />
				<span class="description"><?php _e( 'Text to be displayed on your button link.', 'tjdeals' ); ?></span>
			</label>
		</div>

		<div class="tjd-input">
			<input type="text" name="tjd-button-text" id="tjd-button-text" value="<?php echo esc_attr( get_post_meta( $post->ID, 'tjd_button_text', true ) ); ?>" size="30" style="width: 99%;" placeholder="<?php esc_attr_e( 'Visit Theme Junkie &rarr;', 'tjdeals' ); ?>" />
		</div>

	</div><!-- #tjd-block -->

	<?php
}

/**
 * Saves the metadata for the team item info meta box.
 *
 * @param  int     $post_id
 * @param  object  $post
 * @since  0.1.0
 * @access public
 */
function tjd_meta_boxes_save( $post_id, $post ) {

	if ( ! isset( $_POST['tjd-metaboxes-deal-nonce'] ) || ! wp_verify_nonce( $_POST['tjd-metaboxes-deal-nonce'], basename( __FILE__ ) ) )
		return;

	if ( ! current_user_can( 'edit_post', $post_id ) )
		return;

	$meta = array(
		'tjd_coupon_code' => wp_filter_post_kses( $_POST['tjd-coupon-code'] ),
		'tjd_custom_url'  => esc_url( $_POST['tjd-custom-url'] ),
		'tjd_button_text' => wp_filter_post_kses( $_POST['tjd-button-text'] )
	);

	foreach ( $meta as $meta_key => $new_meta_value ) {

		/* Get the meta value of the custom field key. */
		$meta_value = get_post_meta( $post_id, $meta_key, true );

		/* If there is no new meta value but an old value exists, delete it. */
		if ( current_user_can( 'delete_post_meta', $post_id, $meta_key ) && '' == $new_meta_value && $meta_value )
			delete_post_meta( $post_id, $meta_key, $meta_value );

		/* If a new meta value was added and there was no previous value, add it. */
		elseif ( current_user_can( 'add_post_meta', $post_id, $meta_key ) && $new_meta_value && '' == $meta_value )
			add_post_meta( $post_id, $meta_key, $new_meta_value, true );

		/* If the new meta value does not match the old value, update it. */
		elseif ( current_user_can( 'edit_post_meta', $post_id, $meta_key ) && $new_meta_value && $new_meta_value != $meta_value )
			update_post_meta( $post_id, $meta_key, $new_meta_value );
	}

}