<?php

namespace BeycanPress\Walogin;

use \Beycan\Response;
use \BeycanPress\Walogin\Entity\User;

class Api extends PluginHero\Api
{
    private $errorCodes = [
        1 => 'MEMBERSHIP_NOT_FOUND',
        2 => 'REGISTER',
    ];

    public function __construct()
    {
        User::$currentUserId = get_current_user_id();
        
        $this->addRoutes([
            'walogin-api' => [
                'login' => [
                    'callback' => 'login',
                    'methods' => ['POST']
                ],
                'register' => [
                    'callback' => 'register',
                    'methods' => ['POST']
                ],
                'get-sign-message' => [
                    'callback' => 'getSignMessage',
                    'methods' => ['POST']
                ],
                'matching-control' => [
                    'callback' => 'matchingControl',
                    'methods' => ['POST']
                ],
                'remove-matching' => [
                    'callback' => 'removeMatching',
                    'methods' => ['POST']
                ],
                'address-match' => [
                    'callback' => 'addressMatch',
                    'methods' => ['POST']
                ],
                'address-change' => [
                    'callback' => 'addressChange',
                    'methods' => ['POST']
                ],
            ]
        ]);
    }

    /**
     * @param WP_REST_Request $request
     * @return void
     */
    public function getSignMessage($request) : void
    {
        try {
            $user = new User($request->get_param('address'));
        } catch (\Exception $e) {
            Response::error($e->getMessage());
        }

        Response::success($user->getSignMessage());
    }

    /**
     * @param WP_REST_Request $request
     * @return void
     */
    public function matchingControl($request) : void
    {
        try {
            $user = new User($request->get_param('address'));
            $wlUser = $user->getWlUser();

            if ($wlUser && $wlUser->walletAddress == $request->get_param('address')) {
                Response::error(esc_html__('This account is already paired with the wallet address below!', 'walogin'), [
                    'address' => $wlUser->walletAddress
                ], 'ALREADY_MATCHING');
            } elseif ($wlUser && $wlUser->walletAddress != $request->get_param('address')) {
                Response::error(esc_html__('Are you sure you want to replace the address below with the new address?', 'walogin'), [
                    'oldAddress' => $wlUser->walletAddress,
                    'newAddress' => $request->get_param('address')
                ], 'ADDRESS_CHANGE');
            } 
            
            $wlUser = (new Models\User())->findOneBy(['walletAddress' => $request->get_param('address')]);
            if ($wlUser && $wlUser->userId != User::$currentUserId) {
                Response::error(esc_html__('This wallet address has already been matched with another account.', 'walogin'), null, 'MATCHING_ANOTHER_ACCOUNT');
            }
        } catch (\Exception $e) {
            Response::error($e->getMessage());
        }

        Response::success();
    }

    /**
     * @param WP_REST_Request $request
     * @return void
     */
	public function removeMatching($request) : void
    {
        $address = $request->get_param('address');
        $signature = $request->get_param('signature');

        $this->signatureValidateProcess($address, $signature, function($user) {
            $user->delete();
        });

        Response::success(esc_html__('The matching has been successfully removed!', 'walogin'));
    }

    /**
     * @param WP_REST_Request $request
     * @return void
     */
	public function addressMatch($request) : void
    {
        $address = $request->get_param('address');
        $signature = $request->get_param('signature');

        $this->signatureValidateProcess($address, $signature, function($user) {
            $user->insert();
        });

        Response::success(esc_html__('The matching has been successfully completed!', 'walogin'));
    }

    /**
     * @param WP_REST_Request $request
     * @return void
     */
	public function addressChange($request) : void
    {
        $address = $request->get_param('address');
        $signature = $request->get_param('signature');

        $this->signatureValidateProcess($address, $signature, function($user){
            $user->change();
        });

        Response::success(esc_html__('The address change has been successfully completed!', 'walogin'));
    }

    /**
     * @param WP_REST_Request $request
     * @return void
     */
	public function login($request) : void
    {
        User::$currentUserId = 0;
        $address = $request->get_param('address');
        $signature = $request->get_param('signature');
        $redirectTo = $this->getRedirectUrl($request->get_param('redirectTo'));

        $this->signatureValidateProcess($address, $signature, function($user) {
            $user->login();
        });

        Response::success(esc_html__('Successfully logged in!', 'walogin'), compact('redirectTo'));
	}

    /**
     * @param WP_REST_Request $request
     * @return void
     */
	public function register($request) : void
    {
        User::$currentUserId = 0;
        $email = $request->get_param('email');
        $username = $request->get_param('username');
        $address = $request->get_param('address');
        $signature = $request->get_param('signature');
        $redirectTo = $this->getRedirectUrl($request->get_param('redirectTo'));

        if (!$email || !$username) {
            Response::error(esc_html__('Please fill out the required fields.', 'walogin'));
        }

        if (is_email($email) == false) {
            Response::error(esc_html__('You entered an invalid email address!', 'walogin'));
        }

        if (validate_username($username) == false) {
            Response::error(esc_html__('You entered an invalid username!', 'walogin'));
        }

		if ((new Models\User())->findOneBy(['walletAddress' => $address])) {
            Response::error(esc_html__('This wallet address is already paired with an account!', 'walogin'));
        }

        if (email_exists($email)) {
            Response::error(esc_html__('This email address is already registered!', 'walogin'));
        } elseif (username_exists($username)) {
            Response::error(esc_html__('This username is already registered!', 'walogin'));
        }

        $this->signatureValidateProcess($address, $signature, function($user) use ($username, $email) {
            $user->registerAndLogin($username, $email);
        });

        Response::success(esc_html__('Successfully registered & logged in! Your password has been sent to your email address.', 'walogin'), compact('redirectTo'));
	}

    /**
     * 
     * @param string $address
     * @param string $signature
     * @param callable $callback
     * @return void
     */
    private function signatureValidateProcess(string $address, string $signature, callable $callback) : void
    {
        try {
            $user = new User($address);
        } catch (\Exception $e) {
            Response::error($e->getMessage());
        }

        if (!$user->validateHex($signature)) {
			Response::error(esc_html__('An invalid signature!', 'walogin'));
        }

		if ($user->verifySignature($signature)) {
			try {
                $callback($user);
			} catch (\Exception $e) {
                Response::error($e->getMessage(), null, $this->errorCodes[$e->getCode()]);
			}
		} else {
			Response::error(esc_html__('Signature err, try again.', 'walogin'));
		}
    }

    /**
     * @param string $redirectTo
     * @return string
     */
    private function getRedirectUrl(string $redirectTo) : string
    {
        $redirectTo = $redirectTo ? $redirectTo : admin_url();

        if ($this->setting('loginRedirect')) {
            $redirectTo = $this->setting('loginRedirect') != 'same-page' ? $this->setting('loginRedirect') : $redirectTo;
        }

        return apply_filters("Walogin/RedirectTo", $redirectTo);
    }
}
