<?php

namespace BeycanPress\Walogin\Entity;

use \BeycanPress\Walogin\PluginHero\Helpers;
use \Beycan\SignatureFactory\SignatureFactory;
use \BeycanPress\Walogin\Models\User as UserModel;

class User
{
	use Helpers;

    /**
     * @var object
     */
    private $user;

    /**
     * @var string
     */
    private $signature;

    /**
     * @var object|null
     */
    private $wlUser;

    /**
     * @var string
     */
    private $address;

    /**
     * @var object
     */
    private $userModel;

    /**
     * @var int
     */
    public static $currentUserId;

    /**
     * @param string $address
     */
    public function __construct(string $address)
    {
        if (!$this->validateAddress($address)) {
			throw new \Exception(esc_html__('An invalid wallet address!', 'walogin'));
        }

        $this->address = $address;

		$this->userModel = new UserModel();

		$this->attach();
    }

    /**
     * @param string $address
     * @return bool
     */
    public function validateAddress(string $address) : bool
    {
        return (preg_match('/^0x[a-fA-F0-9]{40}$/', $address) >= 1);
    }

    /**
     * @param string $address
     * @return bool
     */
    public function validateHex(string $address) : bool
    {
        return (preg_match('/^0x[a-fA-F0-9]*$/', $address) >= 1);
    }

    /**
     * @return string
     */
    public function getSignMessage() : string
    {
		$nonce = $this->createNewNonce('_walogin_' . $this->address);

        return 'wl'.wp_hash($nonce . $this->getIp(), 'nonce');
    }

    /**
     * @param string $signature
     * @return boolean
     */
    public function verifySignature(string $signature) : bool
    {
		$message = $this->getSignMessage();

        if (!is_string($address = SignatureFactory::extractAddress($message, $signature))) {
            return false;
        }

        $this->signature = $signature;
        
        return strtolower($this->address) === strtolower($address);
    }
    
    /**
     * @param string $username
     * @param string $email
     * @return void
     */
	public function registerAndLogin($username, $email) : void
	{
        $this->register($username, $email);
		$this->login();
	}

    /**
     * @return object
     */
	public function login() : object
	{
		$this->attach();
        
		if (!$this->wlUser && !get_option('users_can_register')) {
			throw new \Exception(esc_html__('You have not registered on this site; we cannot log you in.', 'walogin'), 1);
		} elseif (!$this->wlUser && get_option('users_can_register')) {
            throw new \Exception(esc_html__('A membership was not found. Do you want to register?', 'walogin'), 2);
        }

		clean_user_cache($this->user->ID);
		wp_clear_auth_cookie();

		wp_set_current_user($this->user->ID);
		wp_set_auth_cookie($this->user->ID, false);
		update_user_caches($this->user);

		$this->userModel->update(
            [
                'loginSignature' => $this->signature,
                'lastLoginDate' => current_time('mysql'),
            ],
            [
                'walletAddress' => $this->address,
                'userId' => $this->user->ID
            ]
        );

		return $this->user;
	}
    
    /**
     * @param string $username
     * @param string $email
     * @return object
     */
	public function register($username, $email) : object
	{
        $password = wp_generate_password();

		if (is_multisite()) {
			$userId = wpmu_create_user($username, $password, $email);
		} else {
			$userId = wp_create_user($username, $password, $email);
		}
        
        if (is_wp_error($userId) || !$userId) {
            throw new \Exception(esc_html__('Error during creation!', 'walogin'));
        }

        $subject = esc_html__('New user registration - Your Password', 'walogin');
        $headers = array('Content-Type: text/html; charset=UTF-8');

        $body = sprintf(
            __("
                Welcome aboard %s<br><br>
                Username: %s<br>
                Password: %s<br>
                Address: %s", 'walogin'
            ),
            $username, $username, $password, $this->address
        );
    
        wp_mail($email, $subject, $body, $headers);

		$this->user = get_user_by('ID', $userId);

		$this->insert();

		return $this->user;
	}

    /**
     * @return void
     */
	private function attach() : void
	{
        $userId = 0;

		$this->wlUser = $this->getWlUser();
        
        if (self::$currentUserId) {
            $userId = self::$currentUserId;
        } elseif (isset($this->wlUser->userId)) {
            $userId = $this->wlUser->userId;
        } 

        $this->user = get_user_by('ID', $userId);
	}

    /**
     * @return void
     */
	public function insert() : void
	{
		$this->userModel->insert([
			'userId' => $this->user->ID,
            'loginSignature' => $this->signature,
			'walletAddress' => $this->address,
            'lastLoginDate'   => current_time('mysql'),
		]);
	}

    /**
     * @return void
     */
	public function delete() : void
	{
		$this->userModel->delete([
			'walletAddress' => $this->address,
			'userId' => $this->user->ID
		]);
	}

    /**
     * @return void
     */
	public function change() : void
	{
        $oldWalletAddress = $this->userModel->findOneBy(['userId' => $this->user->ID])->walletAddress;

        if ($this->userModel->findOneBy(['walletAddress' => $this->address])) {
			throw new \Exception(esc_html__('This wallet address has already been paired with another account!', 'walogin'), 1);
		}

		$this->userModel->update(
            [
                'loginSignature' => $this->signature,
                'walletAddress'  => $this->address,
                'matchedDate'    => current_time('mysql'),
            ],
            [
                'walletAddress' => $oldWalletAddress,
                'userId' => $this->user->ID
            ]
        );
	}

    /**
     * @return object|null
     */
    public function getWlUser() : ?object
    {
        if (self::$currentUserId) {
            return $this->userModel->findOneBy(['userId' => self::$currentUserId]);
        } else {
            return $this->userModel->findOneBy(['walletAddress' => $this->address]);
        }
    }
	
}
