<?php

namespace BeycanPress\Walogin;

use \BeycanPress\Walogin\PluginHero\Plugin;
use \BeycanPress\Walogin\PluginHero\Setting;

class Settings extends Setting
{
    use PluginHero\Helpers;

    public function __construct()
    {
        $parent = $this->pages->UserList->slug;
        parent::__construct(esc_html__('Settings', 'walogin'), $parent);
        
        self::createSection(array(
            'id'     => 'generalOptions', 
            'title'  => esc_html__('General options', 'walogin'),
            'icon'   => 'fa fa-cog',
            'fields' => array(
                array(
                    'id'      => 'dds',
                    'title'   => esc_html__('Data deletion status', 'walogin'),
                    'type'    => 'switcher',
                    'default' => false,
                    'help'    => esc_html__('This setting is passive come by default. You enable this setting. All data created by the plug-in will be deleted while removing the plug-in.', 'walogin')
                ),
                array(
                    'id' => 'shortCode',
                    'type' => 'content',
                    'title'   => esc_html__('Login button shortcode', 'walogin'),
                    'content' => '[walogin-init]',
                ),
                array(
                    'id' => 'shortCode2',
                    'type' => 'content',
                    'title'   => esc_html__('Matching button shortcode', 'walogin'),
                    'content' => '[walogin-matching]',
                ),
                array(
                    'id' => 'loginRedirect',
                    'type'  => 'text',
                    'title' => esc_html__('Login redirect url', 'walogin'),
                    'help'  => esc_html__('After logging in, which address do you want to be forwarded to? If you type "same-page" it will redirect to the current page.', 'walogin'),
                    'sanitize' => function($val) {
                        return sanitize_text_field($val);
                    }
                ),
                array(
                    'id' => 'logoutRedirect',
                    'type'  => 'text',
                    'title' => esc_html__('Logout redirect url', 'walogin'),
                    'help'  => esc_html__('After logout, which address do you want to be forwarded to? If you type "same-page" it will redirect to the current page.', 'walogin'),
                    'sanitize' => function($val) {
                        return sanitize_text_field($val);
                    }
                ),
                array(
                    'id'      => 'wpLoginPage',
                    'title'   => esc_html__('WP Login page button', 'walogin'),
                    'type'    => 'switcher',
                    'default' => true,
                    'help'    => esc_html__('Let the wallet login button appear on the WordPress internal login page.', 'walogin')
                ),
                array(
                    'id'      => 'wcLoginPage',
                    'title'   => esc_html__('WC Login page button', 'walogin'),
                    'type'    => 'switcher',
                    'default' => true,
                    'help'    => esc_html__('Let the wallet login button appear on the login page of WooCommerce.', 'walogin')
                ),
                array(
                    'id' => 'infuraProjectId',
                    'type'  => 'text',
                    'title' => esc_html__('Infura Project ID', 'walogin'),
                    'help'  => esc_html__('Please enter an infura project id for WalletConnect to work.', 'walogin'),
                    'sanitize' => function($val) {
                        return sanitize_text_field($val);
                    }
                ),
            )
        ));
        
        self::createSection(array(
            'id'     => 'walletsMenu', 
            'title'  => esc_html__('Accepted wallets', 'walogin'),
            'icon'   => 'fa fa-wallet',
            'fields' => array(
                array(
                    'id'     => 'acceptedWallets',
                    'type'   => 'fieldset',
                    'title'  => esc_html__('Wallets', 'walogin'),
                    'help'   => esc_html__('Specify the wallets you want to accept payments from. (Only payment process)', 'walogin'),
                    'fields' => self::getWallets(),
                    'validate' => function($val) {
                        foreach ($val as $value) {
                            if ($value) {
                                break;
                            } else {
                                return esc_html__('You must activate at least one wallet!', 'walogin');
                            }
                        }
                    }
                ),
            ) 
        ));
        
        if (apply_filters("Walogin/AddonLoad", null)) {
            self::createSection(array(
                'id'     => 'chainsMenu', 
                'title'  => esc_html__('Chains', 'walogin'),
                'icon'   => 'fa fa-link',
                'fields' => array(
                    array(
                        'id'      => 'chains',
                        'title'   => esc_html__('Chains', 'walogin'),
                        'type'    => 'group',
                        'help'    => esc_html__('Activate or add whatever network your NFT is on.', 'walogin'),
                        'button_title' => esc_html__('Add new', 'walogin'),
                        'default' => [
                            [
                                'name' =>  'Main Ethereum Network',
                                'rpcUrl' =>  'https://mainnet.infura.io/v3/9aa3d95b3bc440fa88ea12eaa4456161',
                                'id' =>  1,
                                'explorerUrl' =>  'https://etherscan.io/',
                                'nativeCurrency' => [
                                    'symbol' =>  'ETH',
                                    'decimals' =>  18
                                ]
                            ],
                            [
                                'name' =>  'Binance Smart Chain',
                                'rpcUrl' =>  'https://bsc-dataseed.binance.org/',
                                'id' =>  56,
                                'explorerUrl' =>  'https://bscscan.com/',
                                'nativeCurrency' => [
                                    'symbol' =>  'BNB',
                                    'decimals' =>  18
                                ]
                            ],
                            [
                                'name' =>  'Avalanche Network',
                                'rpcUrl' =>  'https://api.avax.network/ext/bc/C/rpc',
                                'id' =>  43114,
                                'explorerUrl' =>  'https://cchain.explorer.avax.network/',
                                'nativeCurrency' => [
                                    'symbol' =>  'AVAX',
                                    'decimals' =>  18
                                ]
                            ],
                            [
                                'name' =>  'Polygon Mainnet',
                                'rpcUrl' =>  'https://polygon-rpc.com/',
                                'id' =>  137,
                                'explorerUrl' =>  'https://polygonscan.com/',
                                'nativeCurrency' => [
                                    'symbol' =>  'MATIC',
                                    'decimals' =>  18
                                ]
                            ]
                        ],
                        'sanitize' => function($val) {
                            if (is_array($val)) {
                                foreach ($val as $key => &$value) {
                                    $value['name'] = sanitize_text_field($value['name']);
                                    $value['rpcUrl'] = sanitize_text_field($value['rpcUrl']);
                                    $value['id'] = absint($value['id']);
                                    $value['explorerUrl'] = sanitize_text_field($value['explorerUrl']);
                                    $value['nativeCurrency']['symbol'] = strtoupper(sanitize_text_field($value['nativeCurrency']['symbol']));
                                    $value['nativeCurrency']['decimals'] = absint($value['nativeCurrency']['decimals']);
                                }
                            }
    
                            return $val;
                        },
                        'validate' => function($val) {
                            if (is_array($val)) {
                                foreach ($val as $key => $value) {
                                    if (empty($value['name'])) {
                                        return esc_html__('Network name cannot be empty.', 'walogin');
                                    } elseif (empty($value['rpcUrl'])) {
                                        return esc_html__('Network RPC URL cannot be empty.', 'walogin');
                                    } elseif (empty($value['id'])) {
                                        return esc_html__('Chain ID cannot be empty.', 'walogin');
                                    } elseif (empty($value['explorerUrl'])) {
                                        return esc_html__('Explorer URL cannot be empty.', 'walogin');
                                    } elseif (empty($value['nativeCurrency']['symbol'])) {
                                        return esc_html__('Native currency symbol cannot be empty.', 'walogin');
                                    } elseif (empty($value['nativeCurrency']['decimals'])) {
                                        return esc_html__('Native currency Decimals cannot be empty.', 'walogin');
                                    }
                                }
                            } else {
                                return esc_html__('You must add at least one blockchain network!', 'walogin');
                            }
                        },
                        'fields'    => array(
                            array(
                                'title' => esc_html__('Chain name', 'walogin'),
                                'id'    => 'name',
                                'type'  => 'text'
                            ),
                            array(
                                'title' => esc_html__('Chain RPC URL', 'walogin'),
                                'id'    => 'rpcUrl',
                                'type'  => 'text',
                            ),
                            array(
                                'title' => esc_html__('Chain ID', 'walogin'),
                                'id'    => 'id',
                                'type'  => 'number'
                            ),
                            array(
                                'title' => esc_html__('Explorer URL', 'walogin'),
                                'id'    => 'explorerUrl',
                                'type'  => 'text'
                            ),
                            array(
                                'id'     => 'nativeCurrency',
                                'type'   => 'fieldset',
                                'title'  => esc_html__('Native currency', 'walogin'),
                                'fields' => array(
                                    array(
                                        'id'    => 'symbol',
                                        'type'  => 'text',
                                        'title' => esc_html__('Symbol', 'walogin')
                                    ),
                                    array(
                                        'id'    => 'decimals',
                                        'type'  => 'number',
                                        'title' => esc_html__('Decimals', 'walogin')
                                    )
                                ),
                            )
                        ),
                    ),
                ) 
            ));    
        }

        do_action("Walogin/AddonSettings");

        self::createSection(array(
            'id'     => 'backup', 
            'title'  => esc_html__('Backup', 'walogin'),
            'icon'   => 'fa fa-shield',
            'fields' => array(
                array(
                    'type'  => 'backup',
                    'title' => esc_html__('Backup', 'walogin')
                ),
            ) 
        ));

        self::createSection(array(
            'id'     => 'addOns', 
            'title'  => esc_html__('Add-ons', 'walogin'),
            'icon'   => 'fas fa-puzzle-piece',
            'fields' => array(
                array(
                    'id' => 'restrictNFT',
                    'type' => 'content',
                    'content' => '<a href="https://1.envato.market/6bjoqN" target="_blank"><img src="https://i.hizliresim.com/7yc3mnw.png"></a>',
                    'title' => esc_html__('Restrict Content By NFT', 'walogin')
                ),
                array(
                    'id' => 'restrictToken',
                    'type' => 'content',
                    'content' => '<a href="https://1.envato.market/RyQma7" target="_blank"><img src="https://i.hizliresim.com/mtrrhid.png"></a>',
                    'title' => esc_html__('Restrict Content By Token', 'walogin')
                ),
            ) 
        ));
    }

    public static function getAcceptedWallets() : array
    {
		$acceptedWallets = Plugin::$instance->setting('acceptedWallets');
		
		if (!$acceptedWallets) return [];
		
        $acceptedWallets = array_filter($acceptedWallets, function($val) {
            return $val;
        });

        return array_keys($acceptedWallets);
    }

    public static function getWallets() : array
    {
		return array(
            array(
                'id'      => 'metamask',
                'title'   => esc_html('MetaMask'),
                'type'    => 'switcher',
                'default' => true,
            ),
            array(
                'id'      => 'trustwallet',
                'title'   => esc_html('Trust Wallet'),
                'type'    => 'switcher',
                'default' => true,
            ),
            array(
                'id'      => 'binancewallet',
                'title'   => esc_html('Binance Wallet'),
                'type'    => 'switcher',
                'default' => true,
            ),
            array(
                'id'      => 'walletconnect',
                'title'   => esc_html('WalletConnect'),
                'type'    => 'switcher',
                'default' => true,
            ),
        );
    }

    public static function getWalletNames()
    {
        $walletNames = [];

        foreach (self::getWallets() as $value) {
            $walletNames[$value['id']] = $value['title'];
        }

        return $walletNames;
    }
}