(($) => {

    function disableScreen() {
        var div = document.createElement("div");
        div.className += "overlay";
        div.style.backgroundColor = "#EFEFEF";
        div.style.position = "fixed";
        div.style.width = "100%";
        div.style.height = "100%";
        div.style.zIndex = "999999999999999";
        div.style.top = "0px";
        div.style.left = "0px";
        div.style.opacity = ".5";
        document.body.appendChild(div);
    }

    function enableScreen() {
        if (document.querySelector(".overlay")) {
            document.querySelector(".overlay").remove();
        }
    }

    function swalPopup(message, type, html = null) {
        enableScreen();
        return Swal.fire({
            title: message,
            html,
            icon: type,
            didOpen: () => {
                Swal.hideLoading();
            }
        });
    }
    function infoPopup(message, html = null) {
        return swalPopup(message, 'info', html);
    }
    
    function errorPopup(message, html = null) {
        return swalPopup(message, 'error', html);
    }
    
    function successPopup(message, html = null) {
        return swalPopup(message, 'success', html);
    }
    
    function waitingPopup(title, html = null) {
        Swal.fire({
            title,
            html,
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
    }

    Walogin.infoPopup = infoPopup;
    Walogin.errorPopup = errorPopup;
    Walogin.successPopup = successPopup;
    Walogin.waitingPopup = waitingPopup;
    Walogin.disableScreen = disableScreen;
    Walogin.enableScreen = enableScreen;

    function errorMessages(error) {
        if (typeof error === 'object') {
            if (error.code == 4001 || error.message == 'An unexpected error occurred') {
                infoPopup(Walogin.lang.connectionRefused);
            } else if (error.message != 'User closed modal') {
                errorPopup(error.message);
            }
        } else if (error == 'connection-failed') {
            errorPopup(Walogin.lang.connectionFailed);
        } else if (error == 'wallet-not-detected') {
            infoPopup(Walogin.lang.notDetected);
        } else if (error == 'not-found-infura-id') {
            infoPopup(Walogin.lang.notFoundInfuraId);
        } else if (error == 'already-processing') {
            infoPopup(Walogin.lang.alreadyProcessing);
        } else {
            errorPopup(error.message);
        }
    }

    localStorage.removeItem("walletconnect");

    const multiChain = new MultiChain({
        acceptedWallets: Walogin.acceptedWallets,
        allowedNetworks: true,
        infuraId: Walogin.infuraId
    });

    Walogin.multiChain = multiChain;

    // Hooks
    Walogin.hooks = [];

    Walogin.addHook = (hook, callback, priority = 10) => {
        if (typeof callback === 'function') {
            if (typeof Walogin.hooks[hook] === 'undefined') {
                Walogin.hooks[hook] = [];
            }
            Walogin.hooks[hook].push({priority, callback});
            Walogin.hooks[hook].sort((a, b) => a.priority - b.priority);
        } else {
            throw new Error('Callback must be a function');
        }

        return true;
    };

    Walogin.callHook = (hook, ...params) => {
        if (typeof Walogin.hooks[hook] !== 'undefined') {
            Walogin.hooks[hook].forEach((hook) => {
                hook.callback(...params);
            });
        } else {
            throw new Error('Not found hook');
        }

        return true;
    };

    Walogin.removeHook = (hook) => {
        if (typeof Walogin.hooks[hook] !== 'undefined') {
            delete Walogin.hooks[hook];
        } else {
            throw new Error('Not found hook');
        }

        return true;
    };
    
    function openModal(e) {
        e.preventDefault();
        $(".connector-modal").attr("data-type", $(e.target).attr("data-type"));
        $(".connector-modal").addClass("open");
    }

    function closeModal() {
        $(".connector-modal").removeClass("open");
    }

    function updateCookie(key, val) {
        let cookie = $.cookie('wl-connected') || {};
        cookie = typeof cookie !== 'object' ? JSON.parse(cookie) : cookie;
        cookie[key] = val;
        $.cookie('wl-connected', JSON.stringify(cookie), {path: '/'});
    }

    function getCookie(key = null) {
        let cookie = $.cookie('wl-connected') || {};
        cookie = typeof cookie !== 'object' ? JSON.parse(cookie) : cookie;
        if (!key) return cookie;
        return cookie[key] || null;
    }

    // events

    $(document).on('click', ".walogin", openModal);

    window.addEventListener('click', (e) => {
        if (e.target == $(".connector-modal")[0]) {
            closeModal();
        }
    });    

    $(document).on('click', ".wallet", function() {
        let wallet = $(this).attr("data-wallet");
        let type = $(".connector-modal").attr("data-type");

        if (wallet != 'walletconnect') {
            waitingPopup(Walogin.lang.waitingConnection);
        }

        multiChain.connect(wallet)
        .then((connectedAccount) => {
            closeModal();
            setTimeout(() => {
                updateCookie('wallet', wallet);
                updateCookie('address', connectedAccount);
                if (type == 'only-connect-wallet') {
                    Swal.close();
                    $(".walogin").addClass('hidden');
                    $(".walogin-process").removeClass('hidden');
                } else if (type == 'matching') {
                    matchingProcess(connectedAccount);
                } else if (type == 'change-address') {
                    addressChangeProcess(connectedAccount);
                } else if (type == 'remove-matching') {
                    removeMatchingProcess(connectedAccount);
                } else if (type == 'login-register') {
                    disableScreen();
                    loginProcess(connectedAccount);
                } else {
                    Swal.close();
                    Walogin.callHook('connectAfter', connectedAccount);
                }
            }, 1000);
        })
        .catch(error => {
            errorMessages(error);
        });
    });

    $(document).on('click', ".walogin-process", async function(e) {
        e.preventDefault();
        
        let type = $(this).attr("data-type");

        if (getCookie('wallet') && !multiChain.connectedAccount) {
            waitingPopup(Walogin.lang.waitingConnection);
            await multiChain.connect(getCookie('wallet'));
        } else if (!getCookie('wallet')) {
            return openModal(e);
        }

        if (type == 'matching') {
            matchingProcess(multiChain.connectedAccount);
        } else if (type == 'change-address') {
            addressChangeProcess(multiChain.connectedAccount);
        } else if (type == 'remove-matching') {
            removeMatchingProcess(multiChain.connectedAccount);
        }
    });

    // sign

    function getSignMessage(address) {
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/get-sign-message',
                data: {
                    address
                },
                success(response) {
                    if (response.success) {
                        resolve(response.message);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function signProcess(address) {
        return new Promise((resolve, reject) => {
            getSignMessage(address)
            .then(message => {
                multiChain.personalSign(message)
                .then(signature => {
                    updateCookie('signature', signature);
                    resolve(signature);
                })
                .catch(error => {
                    if (error == 'signature-request-denied') {
                        infoPopup(Walogin.lang.signatureRequestDenied);
                    } else {
                        errorPopup(Walogin.lang.somethingWentWrong);
                    }
                    reject(error);
                });
            })
            .catch(error => {
                errorPopup(error.message);
                reject(error);
            });
        });
    }

    // login register
    function login(address, signature) {
        let dataRedirect = $(".connector-modal").attr("data-redirect");
        let redirectTo = (new URLSearchParams(window.location.search)).get('redirect_to');
        redirectTo = dataRedirect == 'same-page' ? window.location.href : redirectTo;
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/login',
                data: {
                    address,
                    signature,
                    redirectTo
                },
                beforeSend() {
                    waitingPopup(Walogin.lang.loginPleaseWait);
                },
                success(response) {
                    if (response.success) {
                        try {
                            Walogin.callHook('loginAfter', {
                                counnectedAccount: multiChain.connectedAccount, 
                                response
                            });
                        } catch (error) {
                            successPopup(response.message).then(() => {
                                disableScreen();
                                location.href = response.data.redirectTo;
                            });
                        }
                        resolve(true);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function loginProcess(address) {
        waitingPopup(Walogin.lang.youMustSignToEnter);
        signProcess(address)
        .then(signature => {
            login(address, signature)
            .catch(error => {
                if (error.errorCode == 'MEMBERSHIP_NOT_FOUND') {
                    infoPopup(error.message);
                } else if (error.errorCode == 'REGISTER') {
                    registerProcess(address, signature);
                }
            });
        });
    }

    function register(address, signature, username, email) {
        let redirectTo = (new URLSearchParams(window.location.search)).get('redirect_to') || null;
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/register',
                data: {
                    address,
                    signature,
                    redirectTo,
                    username,
                    email
                },
                success(response) {
                    if (response.success) {
                        resolve(response);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function registerProcess(address, signature) {
        Swal.fire({
            title: Walogin.lang.registerQuestion,
            icon: 'info',
            showCancelButton: true,
            confirmButtonText: Walogin.lang.register
        }).then(async (result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: Walogin.lang.pleaseFillOut,
                    html:
                        '<input type="text" id="wl-username" class="swal2-input" placeholder="'+Walogin.lang.username+'" required>' +
                        '<input type="email" id="wl-email" class="swal2-input" placeholder="'+Walogin.lang.email+'" required>',
                    focusConfirm: false,
                    showCancelButton: true,
                    confirmButtonText: Walogin.lang.register,
                    preConfirm: async () => {
                        
                        let email = document.getElementById('wl-email').value;
                        let username = document.getElementById('wl-username').value;

                        if (!email || !username) {
                            return Swal.showValidationMessage(Walogin.lang.pleaseFillError);
                        }

                        return await register(address, signature, username, email)
                        .catch(error => {
                            Swal.showValidationMessage(error.message);
                        });
                    }
                }).then(async (result) => {
                    if (result.isConfirmed) {
                        try {
                            Walogin.callHook('registerAfter', {
                                counnectedAccount: multiChain.connectedAccount, 
                                response: result.value
                            });
                        } catch (error) {
                            successPopup(result.value.message).then(() => {
                                disableScreen();
                                location.href = result.value.data.redirectTo;
                            });
                        }
                    } else {
                        enableScreen();
                    }
                });
            } else {
                enableScreen();
            }
        });

    }

    // Matching

    function matchingControl(address) {
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/matching-control',
                data: {
                    address
                },
                beforeSend() {
                    waitingPopup(Walogin.lang.pleaseWait);
                },
                success(response) {
                    if (response.success) {
                        resolve(response);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function removeMatching(address, signature) {
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/remove-matching',
                data: {
                    address,
                    signature
                },
                beforeSend() {
                    waitingPopup(Walogin.lang.pleaseWait);
                },
                success(response) {
                    if (response.success) {
                        resolve(response.message);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function addressMatch(address, signature) {
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/address-match',
                data: {
                    address,
                    signature
                },
                beforeSend() {
                    waitingPopup(Walogin.lang.pleaseWait);
                },
                success(response) {
                    if (response.success) {
                        resolve(response.message);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function alreadyMatching(error) {
        Swal.fire({
            title: error.message + "<br><br>" + error.data.address,
            showConfirmButton: false,
            showDenyButton: true,
            showCancelButton: true,
            cancelButtonText: Walogin.lang.cancel,
            denyButtonText: Walogin.lang.removeMatching,
        }).then((result) => {
            if (result.isDenied) {
                removeMatchingProcess(error.data.address);
            } else {
                enableScreen();
            }
        });
    }

    function matchingProcess(address) {
        matchingControl(address)
        .then(() => {
            signProcess(address)
            .then(signature => {
                addressMatch(address, signature)
                .then(message => {
                    successPopup(message).then(() => {
                        disableScreen();
                        window.location.reload();
                    });
                })
                .catch(error => {
                    infoPopup(error.message);
                });
            });
        })
        .catch(error => {
            if (error.errorCode == "ALREADY_MATCHING") {
                alreadyMatching(error);
            } else {
                errorPopup(error.message);
            }
        });
    }
    
    function removeMatchingProcess(address) {
        Swal.fire({
            title: Walogin.lang.removeMatchingQuestion,
            text: Walogin.lang.matchAgain,
            icon: 'info',
            showCancelButton: true,
            confirmButtonText: Walogin.lang.removeMatchingConfirm
        }).then((result) => {
            if (result.isConfirmed) {
                waitingPopup(Walogin.lang.youMustSignToRemove);
                signProcess(address)
                .then(signature => {
                    removeMatching(address, signature)
                    .then((message) => {
                        successPopup(message).then(() => {
                            disableScreen();
                            window.location.reload();
                        });
                    })
                    .catch(error => {
                        errorPopup(error.message);
                    });
                });
            } else {
                enableScreen();
            }
        });
    }

    // Address change

    function addressChange(address, signature) {
        return new Promise((resolve, reject) => {
            $.ajax({
                method: 'POST',
                url: Walogin.apiUrl + '/address-change',
                data: {
                    address,
                    signature
                },
                success(response) {
                    if (response.success) {
                        resolve(response.message);
                    } else {
                        reject(response);
                    }
                },
                error() {
                    reject(Walogin.lang.somethingWentWrong);
                }
            });
        });
    }

    function addressChangeProcess(address) {
        matchingControl(address)
        .catch(error => {
            if (error.errorCode == "ADDRESS_CHANGE") {
                let msg = error.message + "<br><br>" + 
                Walogin.lang.old + ": " + error.data.oldAddress + "<br><br>" + 
                Walogin.lang.new + ": " + error.data.newAddress;
                Swal.fire({
                    title: msg,
                    text: Walogin.lang.changeAgain,
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#3085d6',
                    confirmButtonText: Walogin.lang.changeAddressConfirm
                }).then((result) => {
                    if (result.isConfirmed) {
                        waitingPopup(Walogin.lang.youMustSignToChange);
                        signProcess(address)
                        .then(signature => {
                            addressChange(address, signature)
                            .then((message) => {
                                $(".matching-address").text(address);
                                successPopup(message);
                            })
                            .catch(error => {
                                errorPopup(error.message);
                            });
                        });
                    } else {
                        enableScreen();
                    }
                });
            } else if (error.errorCode == "ALREADY_MATCHING") {
                alreadyMatching(error);
            } else {
                errorPopup(error.message);
            }
        });
    }
})(jQuery);